import os
import sys
import subprocess
import platform
import shutil
from datetime import datetime

# ===================== CONFIG ===================== #

TEMP_VENV = ".anthro_tmp_env"
REQUIRED_PACKAGES = ["pillow"]

FONT_PATH = "Orbitron-VariableFont_wght.ttf"
BRANDING_IMAGE = "branding.png"
OUTPUT_DIR = "output"

BACKGROUND_COLOR = (0, 0, 0)
TEXT_COLOR = (255, 255, 255)

BRANDING_WIDTH_RATIO = 0.35

# ABSOLUTE SAFETY PADDING (pixels)
TOP_BUFFER_PX = 20
BOTTOM_BUFFER_PX = 30
INTER_BLOCK_BUFFER_PX = 20

SUPPORTED_EXTENSIONS = (
    ".png", ".jpg", ".jpeg", ".webp", ".tiff", ".bmp"
)

# ================================================== #
# ---------------- VENV BOOTSTRAP ------------------ #

def is_venv():
    return sys.prefix != sys.base_prefix


def venv_python():
    if platform.system() == "Windows":
        return os.path.join(TEMP_VENV, "Scripts", "python.exe")
    return os.path.join(TEMP_VENV, "bin", "python")


def create_temp_venv():
    subprocess.check_call([sys.executable, "-m", "venv", TEMP_VENV])


def install_packages(python_exe):
    subprocess.check_call([python_exe, "-m", "pip", "install", "--upgrade", "pip"])
    subprocess.check_call([python_exe, "-m", "pip", "install", *REQUIRED_PACKAGES])


def relaunch_self(python_exe):
    subprocess.check_call(
        [python_exe, os.path.abspath(__file__), *sys.argv[1:]]
    )
    sys.exit(0)

# ---------------- STAMP TEXT ---------------------- #

def load_stamp_lines(txt_path):
    timestamp = datetime.utcnow().strftime("%Y-%m-%d %H:%M:%S UTC")
    lines = []

    with open(txt_path, "r", encoding="utf-8") as f:
        for line in f:
            line = line.rstrip()
            if not line:
                continue
            if "[TIMESTAMP]" in line:
                line = line.replace("[TIMESTAMP]", timestamp)
            lines.append(line)

    return lines

# ---------------- FONT FITTING -------------------- #

def compute_best_font(draw, font_path, lines, max_width):
    from PIL import ImageFont

    size = max_width
    while size > 10:
        font = ImageFont.truetype(font_path, size)
        widest = max(draw.textlength(line, font=font) for line in lines)
        if widest <= max_width * 0.95:
            return font
        size -= 2

    return ImageFont.truetype(font_path, 12)


def absolute_text_block_height(font, num_lines):
    ascent, descent = font.getmetrics()
    line_height = ascent + descent
    total_height = num_lines * line_height
    return total_height, line_height

# ---------------- IMAGE PROCESSING ---------------- #

def stamp_images(txt_path):
    from PIL import Image, ImageDraw

    if not os.path.exists(FONT_PATH):
        raise FileNotFoundError(FONT_PATH)
    if not os.path.exists(BRANDING_IMAGE):
        raise FileNotFoundError(BRANDING_IMAGE)

    stamp_lines = load_stamp_lines(txt_path)
    os.makedirs(OUTPUT_DIR, exist_ok=True)

    branding = Image.open(BRANDING_IMAGE).convert("RGBA")

    for filename in os.listdir("."):
        if filename == OUTPUT_DIR:
            continue
        if not filename.lower().endswith(SUPPORTED_EXTENSIONS):
            continue

        src = os.path.abspath(filename)
        dst = os.path.join(OUTPUT_DIR, filename)

        if os.path.exists(dst):
            print(f"⚠ Skipping existing output: {filename}")
            continue

        img = Image.open(src).convert("RGBA")
        w, h = img.size

        draw_tmp = ImageDraw.Draw(img)
        font = compute_best_font(draw_tmp, FONT_PATH, stamp_lines, w)

        # TEXT HEIGHT = lines × exact glyph height
        text_block_h, line_h = absolute_text_block_height(
            font, len(stamp_lines)
        )

        # BRANDING SIZE
        brand_w = int(w * BRANDING_WIDTH_RATIO)
        brand_h = int(branding.height * (brand_w / branding.width))
        branding_resized = branding.resize(
            (brand_w, brand_h), Image.LANCZOS
        )

        # TOTAL EXTRA HEIGHT (ABSOLUTE)
        extra_h = (
            TOP_BUFFER_PX +
            brand_h +
            INTER_BLOCK_BUFFER_PX +
            text_block_h +
            BOTTOM_BUFFER_PX
        )

        canvas = Image.new(
            "RGBA", (w, h + extra_h), BACKGROUND_COLOR
        )
        canvas.paste(img, (0, 0))
        draw = ImageDraw.Draw(canvas)

        # BRANDING PLACEMENT
        y_cursor = h + TOP_BUFFER_PX
        brand_x = (w - brand_w) // 2
        canvas.paste(
            branding_resized, (brand_x, y_cursor), branding_resized
        )

        # TEXT PLACEMENT
        y_cursor += brand_h + INTER_BLOCK_BUFFER_PX
        for i, line in enumerate(stamp_lines):
            text_w = draw.textlength(line, font=font)
            x = (w - text_w) // 2
            y = y_cursor + i * line_h
            draw.text((x, y), line, fill=TEXT_COLOR, font=font)

        canvas.convert("RGB").save(dst, quality=100, subsampling=0)
        print(f"✔ Stamped (absolute layout): {filename}")

# ---------------- MAIN ---------------------------- #

if __name__ == "__main__":
    if len(sys.argv) != 2:
        print("Usage: python STAMP_IMAGES.py stamp.txt")
        sys.exit(1)

    if not is_venv():
        if not os.path.exists(TEMP_VENV):
            create_temp_venv()
        py = venv_python()
        install_packages(py)
        relaunch_self(py)

    try:
        stamp_images(sys.argv[1])
    finally:
        shutil.rmtree(TEMP_VENV, ignore_errors=True)
